<?php
namespace MMP\Menu;

use MMP\Maps_Marker_Pro as MMP;

class Maps extends Menu {
	private $page;

	/**
	 * Sets up the class
	 *
	 * @since 4.26
	 */
	public function __construct() {
		$this->page = null;
	}

	/**
	 * Registers the hooks
	 *
	 * @since 4.0
	 */
	public function init() {
		add_filter('screen_settings', array($this, 'screen_settings'), 10, 2);

		add_action('load-toplevel_page_mapsmarkerpro_maps', array($this, 'load_page'));
		add_action('admin_enqueue_scripts', array($this, 'load_resources'));
		add_action('wp_ajax_mmp_delete_map', array($this, 'delete_map'));
		add_action('wp_ajax_mmp_bulk_action_maps', array($this, 'bulk_action_maps'));
		add_action('wp_ajax_mmp_maps_screen_options', array($this, 'mmp_maps_screen_options'));
	}

	/**
	 * Adds the screen options to the page
	 *
	 * @since 4.8
	 *
	 * @param string $screen_settings Current screen options
	 * @param WP_Screen $screen Current screen object instance
	 */
	public function screen_settings($screen_settings, $screen) {
		$mmp_settings = MMP::get_instance('MMP\Settings');

		if ($screen->id !== 'toplevel_page_mapsmarkerpro_maps') {
			return $screen_settings;
		}

		$options = get_user_meta(get_current_user_id(), 'mapsmarkerpro_maps_options', true);
		$options = $mmp_settings->validate_maps_screen_settings($options);

		ob_start();
		?>
		<fieldset>
			<legend><?= esc_html__('Columns', 'mmp') ?></legend>
			<label><input type="checkbox" class="mmp-maps-column" value="markers" <?= $this->checked(!in_array('markers', $options['hiddenColumns'])) ?>/><?= esc_html__('Markers', 'mmp') ?></label>
			<label><input type="checkbox" class="mmp-maps-column" value="author" <?= $this->checked(!in_array('author', $options['hiddenColumns'])) ?>/><?= esc_html__('Author', 'mmp') ?></label>
			<label><input type="checkbox" class="mmp-maps-column" value="date" <?= $this->checked(!in_array('date', $options['hiddenColumns'])) ?>/><?= esc_html__('Date', 'mmp') ?></label>
			<label><input type="checkbox" class="mmp-maps-column" value="modified" <?= $this->checked(!in_array('modified', $options['hiddenColumns'])) ?>/><?= esc_html__('Modified', 'mmp') ?></label>
			<label><input type="checkbox" class="mmp-maps-column" value="used_in" <?= $this->checked(!in_array('used_in', $options['hiddenColumns'])) ?>/><?= esc_html__('Used in content', 'mmp') ?></label>
			<label><input type="checkbox" class="mmp-maps-column" value="shortcode" <?= $this->checked(!in_array('shortcode', $options['hiddenColumns'])) ?>/><?= esc_html__('Shortcode', 'mmp') ?></label>
		</fieldset>
		<fieldset>
			<legend><?= esc_html__('Pagination', 'mmp') ?></legend>
			<label for="maps-per-page"><?= esc_html__('Maps per page', 'mmp') ?>:</label>
			<input type="number" id="maps-per-page" min="1" max="1000" step="1" value="<?= $options['perPage'] ?>" />
		</fieldset>
		<p><button type="submit" class="button button-primary"><?= esc_html__('Apply', 'mmp') ?></button></p>
		<?php
		$screen_settings .= ob_get_clean();

		return $screen_settings;
	}

	/**
	 * Executes when the page is loaded
	 *
	 * @since 4.26
	 */
	public function load_page() {
		$db = MMP::get_instance('MMP\DB');
		$l10n = MMP::get_instance('MMP\L10n');
		$api = MMP::get_instance('MMP\API');
		$mmp_settings = MMP::get_instance('MMP\Settings');

		$args = array(
			'page'      => 'mapsmarkerpro_maps',
			'search'    => (isset($_GET['search'])) ? wp_unslash($_GET['search']) : '',
			'orderby'   => (isset($_GET['orderby'])) ? wp_unslash($_GET['orderby']) : 'id',
			'sortorder' => (isset($_GET['sortorder']) && $_GET['sortorder'] === 'desc') ? 'desc' : 'asc',
			'paged'     => (isset($_GET['paged'])) ? absint($_GET['paged']) : 1
		);

		$options = get_user_meta(get_current_user_id(), 'mapsmarkerpro_maps_options', true);
		$options = $mmp_settings->validate_maps_screen_settings($options);

		$map_count = $db->count_maps(array(
			'name' => $args['search']
		));

		$page_count = ceil($map_count / $options['perPage']);
		$args['paged'] = ($args['paged'] < 1) ? 1 : $args['paged'];
		$args['paged'] = ($args['paged'] > $page_count) ? $page_count : $args['paged'];

		$query = http_build_query(array_filter($args), '', '&');
		$url = get_admin_url(null, "admin.php?$query");

		if (!headers_sent() && !$this->compare_page_args($args)) {
			wp_redirect($url);
			die;
		}

		$count = !in_array('markers', $options['hiddenColumns']);
		$columns = array('id', 'name', 'settings', 'author_id');
		if (!in_array('author', $options['hiddenColumns'])) {
			$columns[] = 'author';
		}
		if (!in_array('date', $options['hiddenColumns'])) {
			$columns[] = 'date';
		}
		if (!in_array('modified', $options['hiddenColumns'])) {
			$columns[] = 'modified';
		}
		$maps = $db->get_all_maps($count, array(
			'columns'   => $columns,
			'name'      => $args['search'],
			'orderby'   => $args['orderby'],
			'sortorder' => $args['sortorder'],
			'offset'    => ($args['paged'] - 1) * $options['perPage'],
			'limit'     => $options['perPage']
		));

		$shortcodes = array();
		foreach ($maps as $key => $map) {
			if (!in_array('used_in', $options['hiddenColumns'])) {
				$shortcodes[$map->id] = $db->get_map_shortcodes($map->id);
			}
			$maps[$key]->settings = json_decode($map->settings);
		}

		$bulk_maps = $db->get_all_maps(false, array(
			'columns' => array('id', 'name')
		));

		ob_start();
		?>
		<div class="wrap">
			<h1 class="wp-heading-inline"><?= esc_html__('Maps', 'mmp') ?></h1>
			<a href="<?= get_admin_url(null, 'admin.php?page=mapsmarkerpro_map') ?>" class="page-title-action"><?= esc_html__('Add new map', 'mmp') ?></a>
			<form id="maps_filter" method="GET">
				<input type="hidden" name="page" value="<?= $args['page'] ?>" />
				<input type="hidden" name="orderby" value="<?= esc_attr($args['orderby']) ?>" />
				<input type="hidden" name="sortorder" value="<?= $args['sortorder'] ?>" />
				<div class="mmp-tablenav mmp-tablenav-top mmp-tablenav-maps">
					<div><?= esc_html__('Total maps:', 'mmp') ?> <?= $map_count ?></div>
					<div>
						<input type="text" class="mmp-search" name="search" value="<?= esc_attr($args['search']) ?>" />
						<button type="submit" class="button"><?= esc_html__('Search maps', 'mmp') ?></button>
					</div>
					<div class="mmp-tablenav-pagination">
						<?php if ($args['paged'] < 2): ?>
							<span class="button disabled">&laquo;</span>
							<span class="button disabled">&lsaquo;</span>
						<?php else: ?>
							<a class="button" href="<?= add_query_arg('paged', 1) ?>">&laquo;</a>
							<a class="button" href="<?= add_query_arg('paged', $args['paged'] - 1) ?>">&lsaquo;</a>
						<?php endif; ?>
						<span><input class="mmp-page" type="text" name="paged" value="<?= $args['paged'] ?>" /> <?= esc_html__('of', 'mmp') ?> <?= $page_count ?></span>
						<?php if ($args['paged'] >= $page_count): ?>
							<span class="button disabled">&rsaquo;</span>
							<span class="button disabled">&raquo;</span>
						<?php else: ?>
							<a class="button" href="<?= add_query_arg('paged', $args['paged'] + 1) ?>">&rsaquo;</a>
							<a class="button" href="<?= add_query_arg('paged', $page_count) ?>">&raquo;</a>
						<?php endif; ?>
					</div>
				</div>
			</form>
			<table class="mmp-table mmp-table-maps">
				<tr>
					<th><input type="checkbox" id="select-all" /></th>
					<th><a href="<?= $this->get_column_sort_link('id', $args) ?>" class="mmp-sortable" title="<?= esc_html__('click to sort', 'mmp') ?>"><?= esc_html__('ID', 'mmp') ?></a></th>
					<th><a href="<?= $this->get_column_sort_link('name', $args) ?>" class="mmp-sortable" title="<?= esc_html__('click to sort', 'mmp') ?>"><?= esc_html__('Name', 'mmp') ?></a></th>
					<?php if (!in_array('markers', $options['hiddenColumns'])): ?>
						<th><?= esc_html__('Markers', 'mmp') ?></th>
					<?php endif; ?>
					<?php if (!in_array('author', $options['hiddenColumns'])): ?>
						<th><a href="<?= $this->get_column_sort_link('author', $args) ?>" class="mmp-sortable" title="<?= esc_html__('click to sort', 'mmp') ?>"><?= esc_html__('Author', 'mmp') ?></a></th>
					<?php endif; ?>
					<?php if (!in_array('date', $options['hiddenColumns'])): ?>
						<th><a href="<?= $this->get_column_sort_link('date', $args) ?>" class="mmp-sortable" title="<?= esc_html__('click to sort', 'mmp') ?>"><?= esc_html__('Date', 'mmp') ?></a></th>
					<?php endif; ?>
					<?php if (!in_array('modified', $options['hiddenColumns'])): ?>
						<th><a href="<?= $this->get_column_sort_link('modified', $args) ?>" class="mmp-sortable" title="<?= esc_html__('click to sort', 'mmp') ?>"><?= esc_html__('Modified', 'mmp') ?></a></th>
					<?php endif; ?>
					<?php if (!in_array('used_in', $options['hiddenColumns'])): ?>
						<th><?= esc_html__('Used in content', 'mmp') ?></th>
					<?php endif; ?>
					<?php if (!in_array('shortcode', $options['hiddenColumns'])): ?>
						<th><?= esc_html__('Shortcode', 'mmp') ?></th>
					<?php endif; ?>
				</tr>
				<?php if (!count($maps)): ?>
					<tr><td class="mmp-no-results" colspan="<?= 9 - count($options['hiddenColumns']) ?>"><?= esc_html__('No results') ?></td></tr>
				<?php else: ?>
					<?php foreach ($maps as $map): ?>
						<tr>
							<td class="no-width centered"><input type="checkbox" name="selected[]" value="<?= $map->id ?>" /></td>
							<td class="no-width centered"><?= $map->id ?></td>
							<td class="mmp-has-row-actions">
								<?php if ($map->author_id == get_current_user_id() || current_user_can('mmp_edit_other_maps')): ?>
									<a href="<?= get_admin_url(null, 'admin.php?page=mapsmarkerpro_map&id=' . $map->id) ?>" title="<?= esc_html__('Edit map', 'mmp') ?>"><?= ($map->name) ? esc_html($map->name) : esc_html__('(no name)', 'mmp') ?></a>
								<?php else: ?>
									<?= ($map->name) ? esc_html($map->name) : esc_html__('(no name)', 'mmp') ?>
								<?php endif; ?>
								<div class="mmp-row-actions">
									<ul>
										<?php if ($map->author_id == get_current_user_id() || current_user_can('mmp_edit_other_maps')): ?>
											<li><a href="<?= get_admin_url(null, 'admin.php?page=mapsmarkerpro_map&id=' . $map->id) ?>" title="<?= esc_html__('Edit map', 'mmp') ?>"><?= esc_html__('Edit', 'mmp') ?></a></li>
										<?php endif; ?>
										<?php if ($map->author_id == get_current_user_id() || current_user_can('mmp_delete_other_maps')): ?>
											<li><span class="mmp-delete" href="" data-id="<?= $map->id ?>" data-nonce="<?= wp_create_nonce("mmp-delete-map-{$map->id}") ?>" title="<?= esc_html__('Delete map', 'mmp') ?>"><?= esc_html__('Delete', 'mmp') ?></span></li>
										<?php endif; ?>
										<li>
											<?php if ($l10n->check_ml() === 'wpml'): ?>
												<a href="<?= get_admin_url(null, 'admin.php?page=wpml-string-translation%2Fmenu%2Fstring-translation.php&context=Maps+Marker+Pro&search=' . urlencode($map->name)) ?>" target="_blank"><?= esc_html__('Translate', 'mmp') ?></a>
											<?php elseif ($l10n->check_ml() === 'pll'): ?>
												<a href="<?= get_admin_url(null, 'admin.php?page=mlang_strings&s=Map+%28ID+' . $map->id . '%29&group=Maps+Marker+Pro') ?>" target="_blank"><?= esc_html__('Translate', 'mmp') ?></a>
											<?php else: ?>
												<a href="https://www.mapsmarker.com/multilingual/" target="_blank"><?= esc_html__('Translate', 'mmp') ?></a>
											<?php endif; ?>
										</li>
										<li><a href="<?= get_admin_url(null, 'admin.php?page=mapsmarkerpro_marker&basemap=' . $map->settings->basemapDefault . '&lat=' . $map->settings->lat . '&lng=' . $map->settings->lng . '&zoom=' . $map->settings->zoom . '&map=' . $map->id) ?>" target="_blank"><?= esc_html__('Add marker', 'mmp') ?></a></li>
										<?php if (MMP::$settings['apiFullscreen']): ?>
											<li><a href="<?= $api->link("/fullscreen/{$map->id}/") ?>" target="_blank" title="<?= esc_attr__('Open standalone map in fullscreen mode', 'mmp') ?>"><img src="<?= plugins_url('images/icons/fullscreen.svg', MMP::$path) ?>" /></a></li>
											<li><a class="mmp-qrcode-link" href="" data-id="<?= $map->id ?>" data-url="<?= $api->link("/fullscreen/{$map->id}/") ?>" title="<?= esc_attr__('Show QR code for fullscreen map', 'mmp') ?>"><img src="<?= plugins_url('images/icons/qr-code.png', MMP::$path) ?>" /></a></li>
										<?php endif; ?>
										<?php if (MMP::$settings['apiExport']): ?>
											<li><a href="<?= $api->link("/export/geojson/{$map->id}/") ?>" target="_blank" title="<?= esc_attr__('Export as GeoJSON', 'mmp') ?>"><img src="<?= plugins_url('images/icons/geojson.png', MMP::$path) ?>" /></a></li>
											<li><a href="<?= $api->link("/export/kml/{$map->id}/") ?>" target="_blank" title="<?= esc_attr__('Export as KML', 'mmp') ?>"><img src="<?= plugins_url('images/icons/kml.png', MMP::$path) ?>" /></a></li>
											<li><a href="<?= $api->link("/export/georss/{$map->id}/") ?>" target="_blank" title="<?= esc_attr__('Export as GeoRSS', 'mmp') ?>"><img src="<?= plugins_url('images/icons/georss.png', MMP::$path) ?>" /></a></li>
										<?php endif; ?>
									</ul>
									<div id="qrcode_<?= $map->id ?>" class="mmp-qrcode"></div>
								</div>
							</td>
							<?php if (!in_array('markers', $options['hiddenColumns'])): ?>
								<td class="narrow centered">
									<?php if ($map->settings->filtersAllMarkers): ?>
										<?= esc_html__('All', 'mmp') ?> (<?= $map->markers ?>)
									<?php else: ?>
										<?= $map->markers ?>
									<?php endif; ?>
								</td>
							<?php endif; ?>
							<?php if (!in_array('author', $options['hiddenColumns'])): ?>
								<td><?= esc_html($map->author) ?></td>
							<?php endif; ?>
							<?php if (!in_array('date', $options['hiddenColumns'])): ?>
								<td><abbr class="mmp-datetime" title="<?= $l10n->date('datetime', $map->date) ?>"><?= $l10n->date('date', $map->date) ?></abbr></td>
							<?php endif; ?>
							<?php if (!in_array('modified', $options['hiddenColumns'])): ?>
								<td><abbr class="mmp-datetime" title="<?= $l10n->date('datetime', $map->modified) ?>"><?= $l10n->date('date', $map->modified) ?></abbr></td>
							<?php endif; ?>
							<?php if (!in_array('used_in', $options['hiddenColumns'])): ?>
								<td>
									<?php if ($shortcodes[$map->id]): ?>
										<ul class="mmp-used-in">
											<?php foreach ($shortcodes[$map->id] as $shortcode): ?>
												<li>
													<a href="<?= $shortcode['edit'] ?>" title="<?= esc_attr__('Edit post', 'mmp') ?>"><img src="<?= plugins_url('images/icons/edit-layer.png', MMP::$path) ?>" /></a>
													<a href="<?= $shortcode['link'] ?>" title="<?= esc_attr__('View post', 'mmp') ?>"><?= $shortcode['title'] ?></a>
												</li>
											<?php endforeach; ?>
										</ul>
									<?php else: ?>
										<?= esc_html__('Not used in any content', 'mmp') ?>
									<?php endif; ?>
								</td>
							<?php endif; ?>
							<?php if (!in_array('shortcode', $options['hiddenColumns'])): ?>
								<td class="no-width"><input class="mmp-shortcode" type="text" value="[<?= MMP::$settings['shortcode'] ?> map=&quot;<?= $map->id ?>&quot;]" readonly="readonly" /></td>
							<?php endif; ?>
						</tr>
					<?php endforeach; ?>
				<?php endif; ?>
			</table>
			<div class="mmp-tablenav mmp-tablenav-bottom mmp-tablenav-maps">
				<div class="mmp-tablenav-pagination">
					<?php if ($args['paged'] < 2): ?>
						<span class="button disabled">&laquo;</span>
						<span class="button disabled">&lsaquo;</span>
					<?php else: ?>
						<a class="button" href="<?= add_query_arg('paged', 1) ?>">&laquo;</a>
						<a class="button" href="<?= add_query_arg('paged', $args['paged'] - 1) ?>">&lsaquo;</a>
					<?php endif; ?>
					<span><?= $args['paged'] ?> <?= esc_html__('of', 'mmp') ?> <?= $page_count ?></span>
					<?php if ($args['paged'] >= $page_count): ?>
						<span class="button disabled">&rsaquo;</span>
						<span class="button disabled">&raquo;</span>
					<?php else: ?>
						<a class="button" href="<?= add_query_arg('paged', $args['paged'] + 1) ?>">&rsaquo;</a>
						<a class="button" href="<?= add_query_arg('paged', $page_count) ?>">&raquo;</a>
					<?php endif; ?>
				</div>
			</div>
			<?php if (current_user_can('mmp_use_tools')): ?>
				<div class="mmp-bulk">
					<form id="bulk-actions" method="POST">
						<input type="hidden" id="bulk-action-nonce" name="bulk-action-nonce" value="<?= wp_create_nonce('mmp-bulk-action-maps') ?>" />
						<ul>
							<li>
								<label><input type="radio" name="bulk-action" value="duplicate" /><?= esc_html__('Duplicate maps', 'mmp') ?></label>
							</li>
							<li>
								<label><input type="radio" name="bulk-action" value="duplicate-assign" /><?= esc_html__('Duplicate maps and assign the same markers', 'mmp') ?></label>
							</li>
							<li>
								<label><input type="radio" name="bulk-action" value="delete" /><?= esc_html__('Delete maps and unassign markers', 'mmp') ?></label>
							</li>
							<li>
								<label><input type="radio" name="bulk-action" value="delete-assign" /><?= esc_html__('Delete maps and assign markers to this map', 'mmp') ?></label>
								<select id="assign-target" name="assign-target">
									<?php foreach ($bulk_maps as $bulk_map): ?>
										<option value="<?= $bulk_map->id ?>"><?= "[{$bulk_map->id}] " . esc_html($bulk_map->name) ?></option>
									<?php endforeach; ?>
								</select>
							</li>
							<li>
								<label><input type="radio" name="bulk-action" value="delete-delete" /><?= esc_html__('Delete maps and assigned markers', 'mmp') ?></label>
							</li>
						</ul>
						<button type="submit" class="button button-primary"><?= esc_html__('Apply', 'mmp') ?></button>
					</form>
				</div>
			<?php endif; ?>
		</div>
		<?php
		$this->page = ob_get_clean();
	}

	/**
	 * Loads the required resources
	 *
	 * @since 4.0
	 *
	 * @param string $hook The current admin page
	 */
	public function load_resources($hook) {
		if (substr($hook, -strlen('mapsmarkerpro_maps')) !== 'mapsmarkerpro_maps') {
			return;
		}

		$this->load_global_resources($hook);

		wp_enqueue_script('mmp-admin');
		wp_add_inline_script('mmp-admin', 'listMapsActions();');
	}

	/**
	 * Deletes the map
	 *
	 * @since 4.0
	 */
	public function delete_map() {
		$db = MMP::get_instance('MMP\DB');

		$id = (isset($_POST['id'])) ? absint($_POST['id']) : null;
		if (!$id) {
			wp_send_json_error();
		}

		check_ajax_referer("mmp-delete-map-$id", 'nonce');

		$map = $db->get_map($id);
		if (!$map) {
			wp_send_json_error();
		}

		if ($map->author_id != get_current_user_id() && !current_user_can('mmp_delete_other_maps')) {
			wp_send_json_error();
		}

		if (!isset($_POST['con']) || !$_POST['con']) {
			$message = sprintf(esc_html__('Are you sure you want to delete the map with ID %1$s?', 'mmp'), $id) . "\n";

			$shortcodes = $db->get_map_shortcodes($id);
			if (count($shortcodes)) {
				$message .= esc_html__('The map is used in the following content:', 'mmp') . "\n";
				foreach ($shortcodes as $shortcode) {
					$message .= $shortcode['title'] . "\n";
				}
			} else {
				$message .= esc_html__('The map is not used in any content.', 'mmp');
			}

			wp_send_json_success(array(
				'nonce'   => (isset($_POST['nonce'])) ? $_POST['nonce'] : null,
				'id'      => $id,
				'message' => $message
			));
		}

		$delete = $db->delete_map($id);
		if ($delete === false) {
			wp_send_json_error();
		}

		do_action('mmp_delete_map', $id);

		wp_send_json_success(array(
			'id' => $id
		));
	}

	/**
	 * Executes the map bulk actions
	 *
	 * @since 4.0
	 */
	public function bulk_action_maps() {
		global $wpdb;
		$db = MMP::get_instance('MMP\DB');

		check_ajax_referer('mmp-bulk-action-maps', 'nonce');

		if (!current_user_can('mmp_use_tools')) {
			wp_send_json_error();
		}

		$selected = (isset($_POST['selected'])) ? $_POST['selected'] : null;
		$bulk_action = (isset($_POST['bulkAction'])) ? $_POST['bulkAction'] : null;
		$assign_target = (isset($_POST['assignTarget'])) ? absint($_POST['assignTarget']) : null;

		$ids = $db->sanitize_ids($selected);
		if ($bulk_action === 'duplicate') {
			foreach ($ids as $id) {
				$map = $db->get_map($id, false, ARRAY_A);
				if (!$map) {
					continue;
				}
				$insert_id = $db->add_map((object) $map);
				if ($insert_id === false) {
					continue;
				}
				do_action('mmp_save_map', $insert_id, $map, true);
				do_action('mmp_add_map', $insert_id, $map);
			}
		} else if ($bulk_action === 'duplicate-assign') {
			foreach ($ids as $id) {
				$map = $db->get_map($id, false, ARRAY_A);
				if (!$map) {
					continue;
				}
				$insert_id = $db->add_map((object) $map);
				if ($insert_id === false) {
					continue;
				}
				do_action('mmp_save_map', $insert_id, $map, true);
				do_action('mmp_add_map', $insert_id, $map);
				$markers = $db->get_map_markers($id, ARRAY_A);
				if (!count($markers)) {
					continue;
				}
				$assign_ids = array_column($markers, 'id');
				$db->assign_markers($insert_id, $assign_ids);
			}
		} else if ($bulk_action === 'delete') {
			foreach ($ids as $id) {
				$delete = $db->delete_map($id);
				if ($delete === false) {
					continue;
				}
				do_action('mmp_delete_map', $id);
			}
		} else if ($bulk_action === 'delete-assign') {
			foreach ($ids as $id) {
				$markers = $db->get_map_markers($id, ARRAY_A);
				$delete = $db->delete_map($id);
				if ($delete === false) {
					continue;
				}
				do_action('mmp_delete_map', $id);
				if (!count($markers)) {
					continue;
				}
				$assign_ids = array_column($markers, 'id');
				$db->assign_markers($assign_target, $assign_ids);
			}
		} else if ($bulk_action === 'delete-delete') {
			foreach ($ids as $id) {
				$markers = $db->get_map_markers($id, ARRAY_A);
				$delete = $db->delete_map($id);
				if ($delete === false) {
					continue;
				}
				do_action('mmp_delete_map', $id);
				foreach ($markers as $marker) {
					$delete = $db->delete_marker($marker['id']);
					if ($delete === false) {
						continue;
					}
					do_action('mmp_delete_marker', $marker['id']);
				}
			}
		}

		wp_send_json_success();
	}

	/**
	 * AJAX request for saving the screen options
	 *
	 * @since 4.8
	 */
	public function mmp_maps_screen_options() {
		$mmp_settings = MMP::get_instance('MMP\Settings');

		check_ajax_referer('screen-options-nonce', 'nonce');

		if (!current_user_can('mmp_view_maps')) {
			wp_send_json_error();
		}

		// Workaround for jQuery not sending empty arrays
		if (!isset($_POST['hiddenColumns'])) {
			$_POST['hiddenColumns'] = array();
		}

		$options = $mmp_settings->validate_maps_screen_settings($_POST, false, false);
		update_user_meta(get_current_user_id(), 'mapsmarkerpro_maps_options', $options);

		wp_send_json_success();
	}

	/**
	 * Shows the maps page
	 *
	 * @since 4.0
	 */
	protected function show() {
		echo $this->page;
	}
}
