<?php
namespace MMP\Sitemap;

use MMP\Maps_Marker_Pro as MMP;

class Sitemap {
	/**
	 * Registers the hooks
	 *
	 * @since 4.29
	 */
	public function init() {
		if (MMP::$settings['sitemapGoogle']) {
			add_action('sm_buildmap', array($this, 'add_sitemap_to_gsg'));
		}
		if (MMP::$settings['sitemapYoast']) {
			add_filter('wpseo_sitemaps_providers', array($this, 'add_sitemap_provider_yoast'));
		}
		if (MMP::$settings['sitemapRankMath']) {
			add_filter('rank_math/sitemap/providers', array($this, 'add_sitemap_provider_rank_math'));
		}
		if (MMP::$settings['sitemapTsf']) {
			add_filter('the_seo_framework_sitemap_additional_urls', array($this, 'add_sitemap_to_tsf'));
		}
	}

	/**
	 * Adds the fullscreen links for maps to the sitemap generated by XML Sitemap Generator for Google
	 *
	 * @since 4.29
	 */
	public function add_sitemap_to_gsg() {
		$db = MMP::get_instance('MMP\DB');
		$api = MMP::get_instance('MMP\API');

		if (!class_exists('GoogleSitemapGenerator')) {
			return;
		}

		if (method_exists('GoogleSitemapGenerator', 'GetInstance') && method_exists('GoogleSitemapGenerator', 'AddUrl')) {
			$get_instance_method = 'GetInstance';
			$add_url_function = 'AddUrl';
		} else if (method_exists('GoogleSitemapGenerator', 'get_instance') && method_exists('GoogleSitemapGenerator', 'add_url')) {
			$get_instance_method = 'get_instance';
			$add_url_function = 'add_url';
		} else {
			return;
		}

		$sitemap = \GoogleSitemapGenerator::$get_instance_method();
		$maps = $db->get_all_maps(false, array(
			'columns' => array('id', 'modified'),
			'include' => MMP::$settings['sitemapGoogleInclude'],
			'exclude' => MMP::$settings['sitemapGoogleExclude']
		));
		foreach ($maps as $map) {
			$sitemap->$add_url_function(
				$api->link("/fullscreen/{$map->id}/"),
				(new \DateTime($map->modified, new \DateTimeZone('UTC')))->getTimestamp(),
				MMP::$settings['sitemapGoogleFrequency'],
				MMP::$settings['sitemapGooglePriority']
			);
		}
	}

	/**
	 * Adds an external sitemap provider to Yoast
	 *
	 * @since 4.29
	 *
	 * @param array $external_providers List of current external providers
	 */
	public function add_sitemap_provider_yoast($external_providers) {
		$external_providers[] = new Provider\Yoast();

		return $external_providers;
	}

	/**
	 * Adds an external sitemap provider to Rank Math
	 *
	 * @since 4.29
	 *
	 * @param array $external_providers List of current external providers
	 */
	public function add_sitemap_provider_rank_math($external_providers) {
		$external_providers[] = new Provider\Rank_Math();

		return $external_providers;
	}

	/**
	 * Adds the fullscreen links for maps to the sitemap generated by The SEO Framework
	 *
	 * @since 4.30
	 */
	public function add_sitemap_to_tsf($custom_urls) {
		$db = MMP::get_instance('MMP\DB');
		$api = MMP::get_instance('MMP\API');

		$maps = $db->get_all_maps(false, array(
			'columns' => array('id', 'modified')
		));
		foreach ($maps as $map) {
			$custom_urls[$api->link("/fullscreen/{$map->id}/")] = array(
				'lastmod' => (new \DateTime($map->modified, new \DateTimeZone('UTC')))->getTimestamp(),
			);
		}

		return $custom_urls;
	}

	/**
	 * Clears the sitemap cache
	 *
	 * @since 4.29
	 */
	public function clear_cache() {
		do_action('sm_rebuild');

		if (class_exists('WPSEO_Sitemaps_Cache_Validator') && method_exists('WPSEO_Sitemaps_Cache_Validator', 'invalidate_storage')) {
			\WPSEO_Sitemaps_Cache_Validator::invalidate_storage();
		}

		if (class_exists('RankMath\Sitemap\Cache') && method_exists('RankMath\Sitemap\Cache', 'invalidate_storage')) {
			\RankMath\Sitemap\Cache::invalidate_storage();
		}

		if (class_exists('The_SEO_Framework\Sitemap\Cache') && method_exists('The_SEO_Framework\Sitemap\Cache', 'clear_sitemap_caches')) {
			\The_SEO_Framework\Sitemap\Cache::clear_sitemap_caches();
		}
	}
}
